using System;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;
using static SupplementaryAreaPlugin.SupplementaryAreasConstants;

namespace SupplementaryAreaPlugin.Actions;

internal interface IDebugModelContext
{
    IAppAccess? App { get; }
    PluginResponse Ok(object shape);
    PluginResponse Fail(int status, string code, string? detail = null);
    string GetQueryValue(System.Collections.Generic.IReadOnlyDictionary<string,string?> query, string key);
}

/// <summary>
/// Encapsulates logic for the "debugModel" GET action.
/// Reads raw stored model JSON and returns lightweight diagnostics (version, area count).
/// </summary>
internal sealed class ActionDebugModel
{
    private readonly IDebugModelContext ctx;
    public ActionDebugModel(IDebugModelContext ctx) => this.ctx = ctx;

    public async Task<PluginResponse> ExecuteAsync(System.Collections.Generic.IReadOnlyDictionary<string,string?> query, CancellationToken ct = default)
    {
        if (ctx.App == null)
            return ctx.Fail(500, ErrorNotInitialised);
        if (!TryGetRecordId(query, out var recordId))
            return ctx.Fail(400, ErrorMissingRecordId);

        string key = ModelKeyPrefix + recordId;
        string? raw = null;
        SupplementaryAreasPlugin.SupplementaryModel? deserialized = null;
        try
        {
            raw = await ctx.App.KeyValue.GetAsync(key, ct);
            if (!string.IsNullOrWhiteSpace(raw))
            {
                try { deserialized = JsonSerializer.Deserialize<SupplementaryAreasPlugin.SupplementaryModel>(raw, JsonOptions); }
                catch (Exception ex)
                {
                    ctx.App.Logger.LogWarning(ex, "debugModel: failed to deserialize raw model for {RecordId}", recordId);
                }
            }
        }
        catch (OperationCanceledException) { throw; }
        catch (Exception ex)
        {
            ctx.App.Logger.LogWarning(ex, "debugModel: failed reading key {Key}", key);
        }

        return ctx.Ok(new
        {
            recordId,
            raw,
            deserializedVersion = deserialized?.Version,
            deserializedAreas = deserialized?.Areas?.Count
        });
    }

    private bool TryGetRecordId(System.Collections.Generic.IReadOnlyDictionary<string,string?> query, out string? recordId)
    {
        recordId = ctx.GetQueryValue(query, QueryRecordId);
        return !string.IsNullOrWhiteSpace(recordId);
    }

    private static readonly JsonSerializerOptions JsonOptions = new(JsonSerializerDefaults.Web) { WriteIndented = false };
}
